"use client";

import { useEffect, useRef } from "react";
import { gsap } from "gsap";
import SpotlightCard from "../../components/SpotlightCard";

interface ServiceCard {
    title: string;
    description: string;
}

const services: ServiceCard[] = [
    {
        title: "Import/Export Expertise",
        description: "Streamlining your global trade needs with integrity and efficiency.",
    },
    {
        title: "Product Sourcing Excellence",
        description: "Connecting you with premium suppliers to ensure top-quality products.",
    },
    {
        title: "Supply Chain Solutions",
        description: "Optimized logistics to guarantee timely delivery and customer satisfaction.",
    },
];

export default function Services() {
    const cardsRef = useRef<(HTMLDivElement | null)[]>([]);

    useEffect(() => {
        const observers: IntersectionObserver[] = [];

        cardsRef.current.forEach((card, index) => {
            if (!card) return;

            gsap.set(card, { y: 100, opacity: 0 });

            const observer = new IntersectionObserver(
                (entries) => {
                    entries.forEach((entry) => {
                        if (entry.isIntersecting) {
                            gsap.to(card, {
                                y: 0,
                                opacity: 1,
                                duration: 0.8,
                                ease: "power3.out",
                                delay: index * 0.1,
                            });
                            observer.unobserve(card);
                        }
                    });
                },
                {
                    threshold: 0.1,
                    rootMargin: "0px 0px -100px 0px",
                }
            );

            observer.observe(card);
            observers.push(observer);
        });

        return () => {
            observers.forEach((observer) => observer.disconnect());
        };
    }, []);

    const setCardRef = (index: number) => (el: HTMLDivElement | null) => {
        cardsRef.current[index] = el;
    };

    return (
        <section id="services" className="w-full py-16 md:py-24 bg-white">
            <div className="max-w-7xl mx-auto">
                <h2 className="text-3xl md:text-4xl lg:text-5xl font-light text-center mb-6 px-5">
                    Reliable Wholesale Distribution Centered on You
                </h2>

                <p className="text-lg md:text-xl text-center text-gray-600 mb-12 md:mb-16 px-5">
                    Explore our comprehensive services that combine quality, transparency, and customer care for seamless trade solutions.
                </p>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 md:gap-8 px-5">
                    {services.map((service, index) => (
                        <div
                            key={index}
                            ref={setCardRef(index)}
                            className="h-full"
                        >
                            <SpotlightCard className="h-full flex flex-col">
                                <h3 className="text-xl md:text-2xl font-light text-white mb-4">
                                    {service.title}
                                </h3>
                                <p className="text-gray-400 text-base md:text-lg leading-relaxed">
                                    {service.description}
                                </p>
                            </SpotlightCard>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}

