"use client";

import { useEffect, useRef } from "react";
import { gsap } from "gsap";
import Image from "next/image";

interface Product {
    title: string;
    description: string;
    image: string;
}

const products: Product[] = [
    {
        title: "Solar Panels & Components",
        description: "We supply premium solar modules and associated components, harnessing renewable energy solutions from trusted international producers. Whether it's for large-scale solar farms or smaller green energy projects, our catalog includes efficient solar panels, inverters, mounting systems, and accessories, all backed by rigorous quality standards.",
        image: "/images/solar.jpeg",
    },
    {
        title: "Chemicals",
        description: "Our chemical supplies include industrial-grade raw materials and specialty chemicals for various use-cases. We ensure that all our chemical products are sourced from certified suppliers and comply with international safety and handling regulations.",
        image: "/images/chemicals.jpeg",
    },
    {
        title: "Home Appliances",
        description: "To elevate everyday living, we import and distribute energy-efficient home appliances. Our range includes refrigerators, washing machines, air conditioners, kitchen gadgets, and smart-home devices — all selected for their performance, longevity, and innovation.",
        image: "/images/appliances.jpeg",
    },
    {
        title: "Bulk Raw Material",
        description: "We provide a wide selection of bulk raw materials essential for manufacturing, industrial production, and large-scale commercial operations. Our sourcing network connects you directly with certified global suppliers, ensuring consistent quality, purity, and cost efficiency across every order. From polymers, metals, and minerals to agricultural and chemical raw inputs, our inventory supports diverse industries with reliable bulk volumes and customizable specifications. Whether you are expanding production capacity or optimizing procurement costs, we deliver raw materials that meet international standards and align seamlessly with your manufacturing requirements.",
        image: "/images/raw.jpeg",
    },
    {
        title: "Poultry Feed",
        description: "Recognizing the critical role of nourishment in poultry farming, we source quality feed blends for large-scale operations. Our global supply network ensures reliable bulk procurement, custom formulations, and consistent delivery, supporting sustainable poultry production.",
        image: "/images/poultry.jpeg",
    },
    {
        title: "Paper Products",
        description: "We supply a diverse range of high-quality paper products designed for commercial, industrial, and everyday use. Our collection includes printing paper, tissue products, kraft paper, packaging sheets, and specialty papers tailored for different manufacturing requirements. Every product is sourced from reputable mills known for consistency, strength, and eco-friendly production practices. Whether you require bulk volumes for retail, packaging operations, offices, or manufacturing lines, we ensure reliable supply, competitive pricing, and consistent quality in every shipment.",
        image: "/images/paper.jpeg",
    },
];

export default function ProductsPage() {
    const cardsRef = useRef<(HTMLDivElement | null)[]>([]);

    useEffect(() => {
        const observers: IntersectionObserver[] = [];

        cardsRef.current.forEach((card, index) => {
            if (!card) return;

            gsap.set(card, { y: 100, opacity: 0 });

            const observer = new IntersectionObserver(
                (entries) => {
                    entries.forEach((entry) => {
                        if (entry.isIntersecting) {
                            gsap.to(card, {
                                y: 0,
                                opacity: 1,
                                duration: 0.8,
                                ease: "power3.out",
                                delay: index * 0.1,
                            });
                            observer.unobserve(card);
                        }
                    });
                },
                {
                    threshold: 0.1,
                    rootMargin: "0px 0px -100px 0px",
                }
            );

            observer.observe(card);
            observers.push(observer);
        });

        return () => {
            observers.forEach((observer) => observer.disconnect());
        };
    }, []);

    const setCardRef = (index: number) => (el: HTMLDivElement | null) => {
        cardsRef.current[index] = el;
    };

    return (
        <section className="w-full py-16 md:py-24 bg-white min-h-screen mt-24">
            <div className="max-w-7xl mx-auto">
                <h2 className="text-3xl md:text-4xl lg:text-5xl font-light text-center mb-6 px-5">
                    Our Products
                </h2>

                <p className="text-lg md:text-xl text-center text-gray-600 mb-12 md:mb-16 px-5">
                    Discover our comprehensive range of products designed to meet your diverse trade and commercial needs.
                </p>

                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 md:gap-8 px-5">
                    {products.map((product, index) => (
                        <div
                            key={index}
                            ref={setCardRef(index)}
                            className="bg-white rounded-lg overflow-hidden shadow-md hover:shadow-lg transition-shadow duration-300 flex flex-col h-full"
                        >
                            <div className="w-full h-64 md:h-80 overflow-hidden bg-gray-200 relative">
                                <Image
                                    src={product.image}
                                    alt={product.title}
                                    fill
                                    className="w-full h-full object-cover"
                                    sizes="(max-width: 768px) 100vw, (max-width: 1024px) 50vw, 33vw"
                                    priority={index < 3}
                                />
                            </div>

                            <div className="p-6 flex-1 flex flex-col">
                                <h3 className="text-xl md:text-2xl font-light text-center mb-4">
                                    {product.title}
                                </h3>
                                <p className="text-gray-600 text-base md:text-lg leading-relaxed text-center flex-1">
                                    {product.description}
                                </p>
                            </div>
                        </div>
                    ))}
                </div>
            </div>
        </section>
    );
}

