"use client";

import { useEffect, useRef } from "react";
import { gsap } from "gsap";
import Image from "next/image";
import Button from "./Button";

interface ProductCard {
    title: string;
    image: string;
}

const products: ProductCard[] = [
    {
        title: "Solar Panel Systems",
        image: "/images/solar.jpeg",
    },
    {
        title: "Industrial Chemical Supplies",
        image: "/images/chemicals.jpeg",
    },
    {
        title: "Bulk Raw Materials",
        image: "/images/raw.jpeg",
    },
];

export default function Products() {
    const cardsRef = useRef<(HTMLDivElement | null)[]>([]);

    useEffect(() => {
        const observers: IntersectionObserver[] = [];

        cardsRef.current.forEach((card, index) => {
            if (!card) return;

            gsap.set(card, { y: 100, opacity: 0 });

            const observer = new IntersectionObserver(
                (entries) => {
                    entries.forEach((entry) => {
                        if (entry.isIntersecting) {
                            gsap.to(card, {
                                y: 0,
                                opacity: 1,
                                duration: 0.8,
                                ease: "power3.out",
                                delay: index * 0.1,
                            });
                            observer.unobserve(card);
                        }
                    });
                },
                {
                    threshold: 0.1,
                    rootMargin: "0px 0px -100px 0px",
                }
            );

            observer.observe(card);
            observers.push(observer);
        });

        return () => {
            observers.forEach((observer) => observer.disconnect());
        };
    }, []);

    const setCardRef = (index: number) => (el: HTMLDivElement | null) => {
        cardsRef.current[index] = el;
    };

    return (
        <section className="w-full py-16 md:py-24 bg-white">
            <div className="max-w-7xl mx-auto">
                <h2 className="text-3xl md:text-4xl lg:text-5xl font-light text-center mb-6 px-5">
                    Trusted Sourcing and Supply Solutions
                </h2>

                <p className="text-lg md:text-xl text-center text-gray-600 mb-12 md:mb-16 px-5">
                    Discover our curated range of products and exclusive offers designed to meet your diverse trade needs.
                </p>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 md:gap-8 px-5">
                    {products.map((product, index) => (
                        <div
                            key={index}
                            ref={setCardRef(index)}
                            className="bg-white rounded-lg overflow-hidden shadow-md hover:shadow-lg transition-shadow duration-300 flex flex-col h-full"
                        >
                            <div className="w-full h-64 md:h-80 overflow-hidden bg-gray-200 relative">
                                <Image
                                    src={product.image}
                                    alt={product.title}
                                    layout="fill"
                                    objectFit="cover"
                                    className="w-full h-full object-cover"
                                    sizes="(max-width: 768px) 100vw, 33vw"
                                    priority={index === 0}
                                />
                            </div>

                            <div className="p-6 flex-1 flex items-center justify-center">
                                <h3 className="text-xl md:text-2xl font-light text-center">
                                    {product.title}
                                </h3>
                            </div>
                        </div>
                    ))}
                </div>
                <div className="flex justify-center mt-16">
                    <Button text="Explore All Products" link="/products" dark={true} />
                </div>
            </div>
        </section>
    );
}
