"use client";

import React from "react";
import { useRouter } from "next/navigation";

interface ButtonProps {
    text: string;
    link: string;
    dark?: boolean;
}

export default function Button({ text, link, dark = true }: ButtonProps) {
    const router = useRouter();
    const baseClasses = "px-6 py-2 rounded-full text-base font-light cursor-pointer transition-all duration-200";
    const darkClasses =
        "bg-black text-white border border-[#222] " +
        "hover:shadow-[0_0_16px_2px_rgba(160,160,160,0.6)] hover:border-[#aaa]";
    const lightClasses =
        "bg-white text-black border border-[#eee] " +
        "hover:shadow-[0_0_16px_2px_rgba(35,35,35,0.18)] hover:border-[#222]";

    const handleClick = (e: React.MouseEvent<HTMLButtonElement>) => {
        e.preventDefault();
        
        // Handle route navigation
        if (link.startsWith('/') && !link.startsWith('/#')) {
            router.push(link);
            return;
        }
        
        // Handle hash links (for scrolling)
        let element: HTMLElement | null = null;
        if (link.startsWith('#') || link.startsWith('/#')) {
            const hash = link.startsWith('/#') ? link.slice(2) : link.slice(1);
            element = document.getElementById(hash);
        } else {
            element = document.getElementById(link);
        }
        
        if (element) {
            const rect = element.getBoundingClientRect();
            const scrollTop = window.pageYOffset || document.documentElement.scrollTop;
            const top = rect.top + scrollTop;

            window.scrollTo({
                top,
                behavior: "smooth"
            });
        }
    };

    return (
        <button
            type="button"
            className={`${baseClasses} ${dark ? darkClasses : lightClasses}`}
            onClick={handleClick}
        >
            {text}
        </button>
    );
}
